//------------------------------------------------------------------------------
// Example software to demonstrate the usage of the HDQ communication library.
// The temperature from an bq27010 gas-gauge IC that is connected via HDQ is
// read out periodically and temperature changes are indicated by operating
// the GPIO pin of the bq27010 device. See application report for more details.
//
// R. Wu
// Texas Instruments Inc.
// January 2010
// Built with IAR Embedded Workbench Version: 4.21A
//------------------------------------------------------------------------------
#include <msp430x20x1.h>
#include "bq27x10.h"
#include "..\..\lib\HDQ\HDQ.h"
//------------------------------------------------------------------------------

#define ATRATE_STEPS           100          // USER CONFIG: # of 3.57 uV steps
#define BUFFERSIZE              2           // # of bytes for Tx & Rx buffers

unsigned char TxData[BUFFERSIZE];           // Stores data bytes to be TX'd
unsigned char RxData[BUFFERSIZE];           // Stores data bytes that are RX'd
unsigned char ctrl;                         // Stores Control Register
unsigned char mode;                         // Stores Mode Register
unsigned char rsoc;                         // Stores Relative State of Charge
unsigned char flags;                        // Stores Flags
unsigned int  temperature;                  // Stores temperature
unsigned int  voltage;                      // Stores voltage
unsigned int  atrate;                       // Stores AtRate
unsigned int  artte;                        // Stores AtRate Time to Empty
  signed int  offset;                       // Stores Offset measurement

unsigned int transBytes2Int(unsigned char msb, unsigned char lsb)
{
  unsigned int tmp;
  
  tmp = ((msb << 8) & 0xFF00);
  return ((unsigned int)(tmp + lsb) & 0x0000FFFF);  
}

void MSP430_bq27010_error(void)
{
  while (1)                                 // Loop forever
  {
    P1OUT ^= BIT0;                          // Toggle LED
    __delay_cycles(500000);                 // Delay
  }
}

void main(void)
{
  unsigned char tmp;
  
  WDTCTL = WDTPW | WDTHOLD;                 // Stop watchdog timer
  P1OUT &= ~BIT0;
  P1SEL &= ~BIT0;                           // P1.x GPIO function
  P1DIR |= BIT0;                            // P1.x output direction

  HDQSetup();                               // Do basic HDQ setup

  __enable_interrupt();                     // Enable global interrupts
 
  // Read Status Flags
  flags = HDQRead(bq27x10CMD_FLAGS);
 
  // GPIO open drain output - Turn ON
  HDQWrite(bq27x10CMD_CTRL, 0xA9);          // Can use any of the 3 command keys
  mode = HDQRead(bq27x10CMD_MODE);
  mode &= ~(bq27x10REG_MODE_GPIEN|bq27x10REG_MODE_GPSTAT);
  HDQWrite(bq27x10CMD_MODE, mode);
  tmp = HDQRead(bq27x10CMD_MODE);
  if (mode != tmp)
  {
    MSP430_bq27010_error();                 // Signal error condition occurred    
  }

  // Read Relative State of Charge (units = %)
  rsoc = HDQRead(bq27x10CMD_RSOC);
 
  // Read Voltage (units = mV)
  RxData[1] = HDQRead(bq27x10CMD_VOLT_MSB);
  RxData[0] = HDQRead(bq27x10CMD_VOLT_LSB);  
  voltage = transBytes2Int(RxData[1], RxData[0]);

  // Read Temperature (units = 0.1K)
  RxData[1] = HDQRead(bq27x10CMD_TEMP_MSB);
  RxData[0] = HDQRead(bq27x10CMD_TEMP_LSB);  
  temperature = transBytes2Int(RxData[1], RxData[0]);
 
  // Set # of steps for AtRate (1 step = 3.57 uV)
  atrate = ATRATE_STEPS;
  HDQWrite(bq27x10CMD_AR_LSB, (atrate & 0x00FF)); // Must set LSB before MSB ???
  HDQWrite(bq27x10CMD_AR_MSB, ((atrate >> 8) & 0x00FF));
  
  // Verify AtRate was set correctly
  RxData[0] = HDQRead(bq27x10CMD_AR_MSB);
  RxData[1] = HDQRead(bq27x10CMD_AR_LSB); 
  if ((transBytes2Int(RxData[0], RxData[1])) != atrate)
  {
    MSP430_bq27010_error();                 // Signal error condition occurred
  }
  
  // Read AtRate Time to Empty (units = Minutes)
  RxData[1] = HDQRead(bq27x10CMD_ARTTE_MSB);
  RxData[0] = HDQRead(bq27x10CMD_ARTTE_LSB); 
  artte = transBytes2Int(RxData[1], RxData[0]);
   
  // Read offset measurement value
  RxData[1] = HDQRead(0x5F);
  RxData[0] = HDQRead(0x5E); 
  offset = transBytes2Int(RxData[1], RxData[0]);

  // Perform internal offset measurement
  HDQWrite(bq27x10CMD_CTRL, 0x56);          // Set up the correct command key
  HDQWrite(bq27x10CMD_MODE, bq27x10REG_MODE_CIO);
  
  // Monitor Calibration-In-Progress flag until offset measurement is done
  while (HDQRead(bq27x10CMD_FLAGS) & bq27x10REG_FLAGS_CALIP);
    
  // Read offset measurement value
  RxData[1] = HDQRead(0x5F);
  RxData[0] = HDQRead(0x5E); 
  offset = transBytes2Int(RxData[1], RxData[0]);
 
  // GPIO open drain output - Turn OFF
  HDQWrite(bq27x10CMD_CTRL, 0xC5);          // Can use any of the 3 command keys
  mode = HDQRead(bq27x10CMD_MODE);
  mode |= (bq27x10REG_MODE_GPIEN|bq27x10REG_MODE_GPSTAT);
  HDQWrite(bq27x10CMD_MODE, mode);
  tmp = HDQRead(bq27x10CMD_MODE);
  if (mode != tmp)
  {
    MSP430_bq27010_error();                 // Signal error condition occurred    
  }  

  // Set P1.0 LED on MSP430 EVM to signal that command sequence was successful
  P1OUT |= BIT0;
}
